/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.collect;

import com.google.common.annotations.GwtCompatible;
import com.google.common.base.Preconditions;
import com.google.common.collect.Ordering;
import com.google.common.math.IntMath;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import org.checkerframework.checker.nullness.qual.Nullable;

@GwtCompatible
final class TopKSelector<T> {
    private final int k;
    private final Comparator<? super T> comparator;
    private final T[] buffer;
    private int bufferSize;
    private @Nullable T threshold;

    public static <T extends Comparable<? super T>> TopKSelector<T> least(int k4) {
        return TopKSelector.least(k4, Ordering.natural());
    }

    public static <T> TopKSelector<T> least(int k4, Comparator<? super T> comparator) {
        return new TopKSelector<T>(comparator, k4);
    }

    public static <T extends Comparable<? super T>> TopKSelector<T> greatest(int k4) {
        return TopKSelector.greatest(k4, Ordering.natural());
    }

    public static <T> TopKSelector<T> greatest(int k4, Comparator<? super T> comparator) {
        return new TopKSelector(Ordering.from(comparator).reverse(), k4);
    }

    private TopKSelector(Comparator<? super T> comparator, int k4) {
        this.comparator = Preconditions.checkNotNull(comparator, "comparator");
        this.k = k4;
        Preconditions.checkArgument(k4 >= 0, "k (%s) must be >= 0", k4);
        Preconditions.checkArgument(k4 <= 0x3FFFFFFF, "k (%s) must be <= Integer.MAX_VALUE / 2", k4);
        this.buffer = new Object[IntMath.checkedMultiply(k4, 2)];
        this.bufferSize = 0;
        this.threshold = null;
    }

    public void offer(@Nullable T elem) {
        if (this.k == 0) {
            return;
        }
        if (this.bufferSize == 0) {
            this.buffer[0] = elem;
            this.threshold = elem;
            this.bufferSize = 1;
        } else if (this.bufferSize < this.k) {
            this.buffer[this.bufferSize++] = elem;
            if (this.comparator.compare(elem, this.threshold) > 0) {
                this.threshold = elem;
            }
        } else if (this.comparator.compare(elem, this.threshold) < 0) {
            this.buffer[this.bufferSize++] = elem;
            if (this.bufferSize == 2 * this.k) {
                this.trim();
            }
        }
    }

    private void trim() {
        int left = 0;
        int right = 2 * this.k - 1;
        int minThresholdPosition = 0;
        int iterations = 0;
        int maxIterations = IntMath.log2(right - left, RoundingMode.CEILING) * 3;
        while (left < right) {
            int pivotIndex = left + right + 1 >>> 1;
            int pivotNewIndex = this.partition(left, right, pivotIndex);
            if (pivotNewIndex > this.k) {
                right = pivotNewIndex - 1;
            } else {
                if (pivotNewIndex >= this.k) break;
                left = Math.max(pivotNewIndex, left + 1);
                minThresholdPosition = pivotNewIndex;
            }
            if (++iterations < maxIterations) continue;
            Arrays.sort(this.buffer, left, right, this.comparator);
            break;
        }
        this.bufferSize = this.k;
        this.threshold = this.buffer[minThresholdPosition];
        for (int i4 = minThresholdPosition + 1; i4 < this.k; ++i4) {
            if (this.comparator.compare(this.buffer[i4], this.threshold) <= 0) continue;
            this.threshold = this.buffer[i4];
        }
    }

    private int partition(int left, int right, int pivotIndex) {
        T pivotValue = this.buffer[pivotIndex];
        this.buffer[pivotIndex] = this.buffer[right];
        int pivotNewIndex = left;
        for (int i4 = left; i4 < right; ++i4) {
            if (this.comparator.compare(this.buffer[i4], pivotValue) >= 0) continue;
            this.swap(pivotNewIndex, i4);
            ++pivotNewIndex;
        }
        this.buffer[right] = this.buffer[pivotNewIndex];
        this.buffer[pivotNewIndex] = pivotValue;
        return pivotNewIndex;
    }

    private void swap(int i4, int j4) {
        T tmp = this.buffer[i4];
        this.buffer[i4] = this.buffer[j4];
        this.buffer[j4] = tmp;
    }

    TopKSelector<T> combine(TopKSelector<T> other) {
        for (int i4 = 0; i4 < other.bufferSize; ++i4) {
            this.offer(other.buffer[i4]);
        }
        return this;
    }

    public void offerAll(Iterable<? extends T> elements) {
        this.offerAll(elements.iterator());
    }

    public void offerAll(Iterator<? extends T> elements) {
        while (elements.hasNext()) {
            this.offer(elements.next());
        }
    }

    public List<T> topK() {
        Arrays.sort(this.buffer, 0, this.bufferSize, this.comparator);
        if (this.bufferSize > this.k) {
            Arrays.fill(this.buffer, this.k, this.buffer.length, null);
            this.bufferSize = this.k;
            this.threshold = this.buffer[this.k - 1];
        }
        return Collections.unmodifiableList(Arrays.asList(Arrays.copyOf(this.buffer, this.bufferSize)));
    }
}

