/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.io;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.base.Preconditions;
import com.google.common.io.ByteArrayDataInput;
import com.google.common.io.ByteArrayDataOutput;
import com.google.common.io.ByteProcessor;
import com.google.common.io.Java8Compatibility;
import com.google.common.math.IntMath;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Queue;

@GwtIncompatible
public final class ByteStreams {
    private static final int BUFFER_SIZE = 8192;
    private static final int ZERO_COPY_CHUNK_SIZE = 524288;
    private static final int MAX_ARRAY_LEN = 0x7FFFFFF7;
    private static final int TO_BYTE_ARRAY_DEQUE_SIZE = 20;
    private static final OutputStream NULL_OUTPUT_STREAM = new OutputStream(){

        @Override
        public void write(int b5) {
        }

        @Override
        public void write(byte[] b5) {
            Preconditions.checkNotNull(b5);
        }

        @Override
        public void write(byte[] b5, int off, int len) {
            Preconditions.checkNotNull(b5);
        }

        public String toString() {
            return "ByteStreams.nullOutputStream()";
        }
    };

    static byte[] createBuffer() {
        return new byte[8192];
    }

    private ByteStreams() {
    }

    @CanIgnoreReturnValue
    public static long copy(InputStream from, OutputStream to) throws IOException {
        int r4;
        Preconditions.checkNotNull(from);
        Preconditions.checkNotNull(to);
        byte[] buf = ByteStreams.createBuffer();
        long total = 0L;
        while ((r4 = from.read(buf)) != -1) {
            to.write(buf, 0, r4);
            total += (long)r4;
        }
        return total;
    }

    @CanIgnoreReturnValue
    public static long copy(ReadableByteChannel from, WritableByteChannel to) throws IOException {
        Preconditions.checkNotNull(from);
        Preconditions.checkNotNull(to);
        if (from instanceof FileChannel) {
            long copied;
            long oldPosition;
            FileChannel sourceChannel = (FileChannel)from;
            long position = oldPosition = sourceChannel.position();
            do {
                copied = sourceChannel.transferTo(position, 524288L, to);
                sourceChannel.position(position += copied);
            } while (copied > 0L || position < sourceChannel.size());
            return position - oldPosition;
        }
        ByteBuffer buf = ByteBuffer.wrap(ByteStreams.createBuffer());
        long total = 0L;
        while (from.read(buf) != -1) {
            Java8Compatibility.flip(buf);
            while (buf.hasRemaining()) {
                total += (long)to.write(buf);
            }
            Java8Compatibility.clear(buf);
        }
        return total;
    }

    private static byte[] toByteArrayInternal(InputStream in, Queue<byte[]> bufs, int totalLen) throws IOException {
        int bufSize = 8192;
        while (totalLen < 0x7FFFFFF7) {
            byte[] buf = new byte[Math.min(bufSize, 0x7FFFFFF7 - totalLen)];
            bufs.add(buf);
            int off = 0;
            while (off < buf.length) {
                int r4 = in.read(buf, off, buf.length - off);
                if (r4 == -1) {
                    return ByteStreams.combineBuffers(bufs, totalLen);
                }
                off += r4;
                totalLen += r4;
            }
            bufSize = IntMath.saturatedMultiply(bufSize, 2);
        }
        if (in.read() == -1) {
            return ByteStreams.combineBuffers(bufs, 0x7FFFFFF7);
        }
        throw new OutOfMemoryError("input is too large to fit in a byte array");
    }

    private static byte[] combineBuffers(Queue<byte[]> bufs, int totalLen) {
        int bytesToCopy;
        byte[] result = new byte[totalLen];
        for (int remaining = totalLen; remaining > 0; remaining -= bytesToCopy) {
            byte[] buf = bufs.remove();
            bytesToCopy = Math.min(remaining, buf.length);
            int resultOffset = totalLen - remaining;
            System.arraycopy(buf, 0, result, resultOffset, bytesToCopy);
        }
        return result;
    }

    public static byte[] toByteArray(InputStream in) throws IOException {
        Preconditions.checkNotNull(in);
        return ByteStreams.toByteArrayInternal(in, new ArrayDeque<byte[]>(20), 0);
    }

    static byte[] toByteArray(InputStream in, long expectedSize) throws IOException {
        int read;
        Preconditions.checkArgument(expectedSize >= 0L, "expectedSize (%s) must be non-negative", expectedSize);
        if (expectedSize > 0x7FFFFFF7L) {
            throw new OutOfMemoryError(new StringBuilder(62).append(expectedSize).append(" bytes is too large to fit in a byte array").toString());
        }
        byte[] bytes = new byte[(int)expectedSize];
        for (int remaining = (int)expectedSize; remaining > 0; remaining -= read) {
            int off = (int)expectedSize - remaining;
            read = in.read(bytes, off, remaining);
            if (read != -1) continue;
            return Arrays.copyOf(bytes, off);
        }
        int b5 = in.read();
        if (b5 == -1) {
            return bytes;
        }
        ArrayDeque<byte[]> bufs = new ArrayDeque<byte[]>(22);
        bufs.add(bytes);
        bufs.add(new byte[]{(byte)b5});
        return ByteStreams.toByteArrayInternal(in, bufs, bytes.length + 1);
    }

    @CanIgnoreReturnValue
    @Beta
    public static long exhaust(InputStream in) throws IOException {
        long read;
        long total = 0L;
        byte[] buf = ByteStreams.createBuffer();
        while ((read = (long)in.read(buf)) != -1L) {
            total += read;
        }
        return total;
    }

    @Beta
    public static ByteArrayDataInput newDataInput(byte[] bytes) {
        return ByteStreams.newDataInput(new ByteArrayInputStream(bytes));
    }

    @Beta
    public static ByteArrayDataInput newDataInput(byte[] bytes, int start) {
        Preconditions.checkPositionIndex(start, bytes.length);
        return ByteStreams.newDataInput(new ByteArrayInputStream(bytes, start, bytes.length - start));
    }

    @Beta
    public static ByteArrayDataInput newDataInput(ByteArrayInputStream byteArrayInputStream) {
        return new ByteArrayDataInputStream(Preconditions.checkNotNull(byteArrayInputStream));
    }

    @Beta
    public static ByteArrayDataOutput newDataOutput() {
        return ByteStreams.newDataOutput(new ByteArrayOutputStream());
    }

    @Beta
    public static ByteArrayDataOutput newDataOutput(int size) {
        if (size < 0) {
            throw new IllegalArgumentException(String.format("Invalid size: %s", size));
        }
        return ByteStreams.newDataOutput(new ByteArrayOutputStream(size));
    }

    @Beta
    public static ByteArrayDataOutput newDataOutput(ByteArrayOutputStream byteArrayOutputStream) {
        return new ByteArrayDataOutputStream(Preconditions.checkNotNull(byteArrayOutputStream));
    }

    @Beta
    public static OutputStream nullOutputStream() {
        return NULL_OUTPUT_STREAM;
    }

    @Beta
    public static InputStream limit(InputStream in, long limit) {
        return new LimitedInputStream(in, limit);
    }

    @Beta
    public static void readFully(InputStream in, byte[] b5) throws IOException {
        ByteStreams.readFully(in, b5, 0, b5.length);
    }

    @Beta
    public static void readFully(InputStream in, byte[] b5, int off, int len) throws IOException {
        int read = ByteStreams.read(in, b5, off, len);
        if (read != len) {
            throw new EOFException(new StringBuilder(81).append("reached end of stream after reading ").append(read).append(" bytes; ").append(len).append(" bytes expected").toString());
        }
    }

    @Beta
    public static void skipFully(InputStream in, long n4) throws IOException {
        long skipped = ByteStreams.skipUpTo(in, n4);
        if (skipped < n4) {
            throw new EOFException(new StringBuilder(100).append("reached end of stream after skipping ").append(skipped).append(" bytes; ").append(n4).append(" bytes expected").toString());
        }
    }

    static long skipUpTo(InputStream in, long n4) throws IOException {
        long totalSkipped;
        long skipped;
        byte[] buf = null;
        for (totalSkipped = 0L; totalSkipped < n4; totalSkipped += skipped) {
            long remaining = n4 - totalSkipped;
            skipped = ByteStreams.skipSafely(in, remaining);
            if (skipped != 0L) continue;
            int skip = (int)Math.min(remaining, 8192L);
            if (buf == null) {
                buf = new byte[skip];
            }
            if ((skipped = (long)in.read(buf, 0, skip)) == -1L) break;
        }
        return totalSkipped;
    }

    private static long skipSafely(InputStream in, long n4) throws IOException {
        int available = in.available();
        return available == 0 ? 0L : in.skip(Math.min((long)available, n4));
    }

    @Beta
    @CanIgnoreReturnValue
    public static <T> T readBytes(InputStream input, ByteProcessor<T> processor) throws IOException {
        int read;
        Preconditions.checkNotNull(input);
        Preconditions.checkNotNull(processor);
        byte[] buf = ByteStreams.createBuffer();
        while ((read = input.read(buf)) != -1 && processor.processBytes(buf, 0, read)) {
        }
        return processor.getResult();
    }

    @Beta
    @CanIgnoreReturnValue
    public static int read(InputStream in, byte[] b5, int off, int len) throws IOException {
        int total;
        int result;
        Preconditions.checkNotNull(in);
        Preconditions.checkNotNull(b5);
        if (len < 0) {
            throw new IndexOutOfBoundsException(String.format("len (%s) cannot be negative", len));
        }
        Preconditions.checkPositionIndexes(off, off + len, b5.length);
        for (total = 0; total < len && (result = in.read(b5, off + total, len - total)) != -1; total += result) {
        }
        return total;
    }

    private static final class LimitedInputStream
    extends FilterInputStream {
        private long left;
        private long mark = -1L;

        LimitedInputStream(InputStream in, long limit) {
            super(in);
            Preconditions.checkNotNull(in);
            Preconditions.checkArgument(limit >= 0L, "limit must be non-negative");
            this.left = limit;
        }

        @Override
        public int available() throws IOException {
            return (int)Math.min((long)this.in.available(), this.left);
        }

        @Override
        public synchronized void mark(int readLimit) {
            this.in.mark(readLimit);
            this.mark = this.left;
        }

        @Override
        public int read() throws IOException {
            if (this.left == 0L) {
                return -1;
            }
            int result = this.in.read();
            if (result != -1) {
                --this.left;
            }
            return result;
        }

        @Override
        public int read(byte[] b5, int off, int len) throws IOException {
            if (this.left == 0L) {
                return -1;
            }
            int result = this.in.read(b5, off, len = (int)Math.min((long)len, this.left));
            if (result != -1) {
                this.left -= (long)result;
            }
            return result;
        }

        @Override
        public synchronized void reset() throws IOException {
            if (!this.in.markSupported()) {
                throw new IOException("Mark not supported");
            }
            if (this.mark == -1L) {
                throw new IOException("Mark not set");
            }
            this.in.reset();
            this.left = this.mark;
        }

        @Override
        public long skip(long n4) throws IOException {
            n4 = Math.min(n4, this.left);
            long skipped = this.in.skip(n4);
            this.left -= skipped;
            return skipped;
        }
    }

    private static class ByteArrayDataOutputStream
    implements ByteArrayDataOutput {
        final DataOutput output;
        final ByteArrayOutputStream byteArrayOutputStream;

        ByteArrayDataOutputStream(ByteArrayOutputStream byteArrayOutputStream) {
            this.byteArrayOutputStream = byteArrayOutputStream;
            this.output = new DataOutputStream(byteArrayOutputStream);
        }

        @Override
        public void write(int b5) {
            try {
                this.output.write(b5);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void write(byte[] b5) {
            try {
                this.output.write(b5);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void write(byte[] b5, int off, int len) {
            try {
                this.output.write(b5, off, len);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeBoolean(boolean v4) {
            try {
                this.output.writeBoolean(v4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeByte(int v4) {
            try {
                this.output.writeByte(v4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeBytes(String s4) {
            try {
                this.output.writeBytes(s4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeChar(int v4) {
            try {
                this.output.writeChar(v4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeChars(String s4) {
            try {
                this.output.writeChars(s4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeDouble(double v4) {
            try {
                this.output.writeDouble(v4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeFloat(float v4) {
            try {
                this.output.writeFloat(v4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeInt(int v4) {
            try {
                this.output.writeInt(v4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeLong(long v4) {
            try {
                this.output.writeLong(v4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeShort(int v4) {
            try {
                this.output.writeShort(v4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeUTF(String s4) {
            try {
                this.output.writeUTF(s4);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public byte[] toByteArray() {
            return this.byteArrayOutputStream.toByteArray();
        }
    }

    private static class ByteArrayDataInputStream
    implements ByteArrayDataInput {
        final DataInput input;

        ByteArrayDataInputStream(ByteArrayInputStream byteArrayInputStream) {
            this.input = new DataInputStream(byteArrayInputStream);
        }

        @Override
        public void readFully(byte[] b5) {
            try {
                this.input.readFully(b5);
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public void readFully(byte[] b5, int off, int len) {
            try {
                this.input.readFully(b5, off, len);
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public int skipBytes(int n4) {
            try {
                return this.input.skipBytes(n4);
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public boolean readBoolean() {
            try {
                return this.input.readBoolean();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public byte readByte() {
            try {
                return this.input.readByte();
            }
            catch (EOFException e5) {
                throw new IllegalStateException(e5);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public int readUnsignedByte() {
            try {
                return this.input.readUnsignedByte();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public short readShort() {
            try {
                return this.input.readShort();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public int readUnsignedShort() {
            try {
                return this.input.readUnsignedShort();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public char readChar() {
            try {
                return this.input.readChar();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public int readInt() {
            try {
                return this.input.readInt();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public long readLong() {
            try {
                return this.input.readLong();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public float readFloat() {
            try {
                return this.input.readFloat();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public double readDouble() {
            try {
                return this.input.readDouble();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public String readLine() {
            try {
                return this.input.readLine();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }

        @Override
        public String readUTF() {
            try {
                return this.input.readUTF();
            }
            catch (IOException e5) {
                throw new IllegalStateException(e5);
            }
        }
    }
}

