/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleMath;
import com.google.common.math.DoubleUtils;
import com.google.common.math.IntMath;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import com.google.common.math.ToDoubleRounder;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;

@GwtCompatible(emulated=true)
public final class BigIntegerMath {
    @VisibleForTesting
    static final int SQRT2_PRECOMPUTE_THRESHOLD = 256;
    @VisibleForTesting
    static final BigInteger SQRT2_PRECOMPUTED_BITS = new BigInteger("16a09e667f3bcc908b2fb1366ea957d3e3adec17512775099da2f590b0667322a", 16);
    private static final double LN_10 = Math.log(10.0);
    private static final double LN_2 = Math.log(2.0);

    @Beta
    public static BigInteger ceilingPowerOfTwo(BigInteger x4) {
        return BigInteger.ZERO.setBit(BigIntegerMath.log2(x4, RoundingMode.CEILING));
    }

    @Beta
    public static BigInteger floorPowerOfTwo(BigInteger x4) {
        return BigInteger.ZERO.setBit(BigIntegerMath.log2(x4, RoundingMode.FLOOR));
    }

    public static boolean isPowerOfTwo(BigInteger x4) {
        Preconditions.checkNotNull(x4);
        return x4.signum() > 0 && x4.getLowestSetBit() == x4.bitLength() - 1;
    }

    public static int log2(BigInteger x4, RoundingMode mode) {
        MathPreconditions.checkPositive("x", Preconditions.checkNotNull(x4));
        int logFloor = x4.bitLength() - 1;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(BigIntegerMath.isPowerOfTwo(x4));
            }
            case DOWN: 
            case FLOOR: {
                return logFloor;
            }
            case UP: 
            case CEILING: {
                return BigIntegerMath.isPowerOfTwo(x4) ? logFloor : logFloor + 1;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                if (logFloor < 256) {
                    BigInteger halfPower = SQRT2_PRECOMPUTED_BITS.shiftRight(256 - logFloor);
                    if (x4.compareTo(halfPower) <= 0) {
                        return logFloor;
                    }
                    return logFloor + 1;
                }
                BigInteger x22 = x4.pow(2);
                int logX2Floor = x22.bitLength() - 1;
                return logX2Floor < 2 * logFloor + 1 ? logFloor : logFloor + 1;
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static int log10(BigInteger x4, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x4);
        if (BigIntegerMath.fitsInLong(x4)) {
            return LongMath.log10(x4.longValue(), mode);
        }
        int approxLog10 = (int)((double)BigIntegerMath.log2(x4, RoundingMode.FLOOR) * LN_2 / LN_10);
        BigInteger approxPow = BigInteger.TEN.pow(approxLog10);
        int approxCmp = approxPow.compareTo(x4);
        if (approxCmp > 0) {
            do {
                --approxLog10;
            } while ((approxCmp = (approxPow = approxPow.divide(BigInteger.TEN)).compareTo(x4)) > 0);
        } else {
            BigInteger nextPow = BigInteger.TEN.multiply(approxPow);
            int nextCmp = nextPow.compareTo(x4);
            while (nextCmp <= 0) {
                ++approxLog10;
                approxPow = nextPow;
                approxCmp = nextCmp;
                nextPow = BigInteger.TEN.multiply(approxPow);
                nextCmp = nextPow.compareTo(x4);
            }
        }
        int floorLog = approxLog10;
        BigInteger floorPow = approxPow;
        int floorCmp = approxCmp;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(floorCmp == 0);
            }
            case DOWN: 
            case FLOOR: {
                return floorLog;
            }
            case UP: 
            case CEILING: {
                return floorPow.equals(x4) ? floorLog : floorLog + 1;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                BigInteger x22 = x4.pow(2);
                BigInteger halfPowerSquared = floorPow.pow(2).multiply(BigInteger.TEN);
                return x22.compareTo(halfPowerSquared) <= 0 ? floorLog : floorLog + 1;
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static BigInteger sqrt(BigInteger x4, RoundingMode mode) {
        MathPreconditions.checkNonNegative("x", x4);
        if (BigIntegerMath.fitsInLong(x4)) {
            return BigInteger.valueOf(LongMath.sqrt(x4.longValue(), mode));
        }
        BigInteger sqrtFloor = BigIntegerMath.sqrtFloor(x4);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(sqrtFloor.pow(2).equals(x4));
            }
            case DOWN: 
            case FLOOR: {
                return sqrtFloor;
            }
            case UP: 
            case CEILING: {
                int sqrtFloorInt = sqrtFloor.intValue();
                boolean sqrtFloorIsExact = sqrtFloorInt * sqrtFloorInt == x4.intValue() && sqrtFloor.pow(2).equals(x4);
                return sqrtFloorIsExact ? sqrtFloor : sqrtFloor.add(BigInteger.ONE);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                BigInteger halfSquare = sqrtFloor.pow(2).add(sqrtFloor);
                return halfSquare.compareTo(x4) >= 0 ? sqrtFloor : sqrtFloor.add(BigInteger.ONE);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    private static BigInteger sqrtFloor(BigInteger x4) {
        BigInteger sqrt0;
        int log2 = BigIntegerMath.log2(x4, RoundingMode.FLOOR);
        if (log2 < 1023) {
            sqrt0 = BigIntegerMath.sqrtApproxWithDoubles(x4);
        } else {
            int shift = log2 - 52 & 0xFFFFFFFE;
            sqrt0 = BigIntegerMath.sqrtApproxWithDoubles(x4.shiftRight(shift)).shiftLeft(shift >> 1);
        }
        BigInteger sqrt1 = sqrt0.add(x4.divide(sqrt0)).shiftRight(1);
        if (sqrt0.equals(sqrt1)) {
            return sqrt0;
        }
        while ((sqrt1 = (sqrt0 = sqrt1).add(x4.divide(sqrt0)).shiftRight(1)).compareTo(sqrt0) < 0) {
        }
        return sqrt0;
    }

    @GwtIncompatible
    private static BigInteger sqrtApproxWithDoubles(BigInteger x4) {
        return DoubleMath.roundToBigInteger(Math.sqrt(DoubleUtils.bigToDouble(x4)), RoundingMode.HALF_EVEN);
    }

    @GwtIncompatible
    public static double roundToDouble(BigInteger x4, RoundingMode mode) {
        return BigIntegerToDoubleRounder.INSTANCE.roundToDouble(x4, mode);
    }

    @GwtIncompatible
    public static BigInteger divide(BigInteger p4, BigInteger q4, RoundingMode mode) {
        BigDecimal pDec = new BigDecimal(p4);
        BigDecimal qDec = new BigDecimal(q4);
        return pDec.divide(qDec, 0, mode).toBigIntegerExact();
    }

    public static BigInteger factorial(int n4) {
        MathPreconditions.checkNonNegative("n", n4);
        if (n4 < LongMath.factorials.length) {
            return BigInteger.valueOf(LongMath.factorials[n4]);
        }
        int approxSize = IntMath.divide(n4 * IntMath.log2(n4, RoundingMode.CEILING), 64, RoundingMode.CEILING);
        ArrayList<BigInteger> bignums = new ArrayList<BigInteger>(approxSize);
        int startingNumber = LongMath.factorials.length;
        long product = LongMath.factorials[startingNumber - 1];
        int shift = Long.numberOfTrailingZeros(product);
        int productBits = LongMath.log2(product >>= shift, RoundingMode.FLOOR) + 1;
        int bits = LongMath.log2(startingNumber, RoundingMode.FLOOR) + 1;
        int nextPowerOfTwo = 1 << bits - 1;
        for (long num = (long)startingNumber; num <= (long)n4; ++num) {
            if ((num & (long)nextPowerOfTwo) != 0L) {
                nextPowerOfTwo <<= 1;
                ++bits;
            }
            int tz = Long.numberOfTrailingZeros(num);
            long normalizedNum = num >> tz;
            shift += tz;
            int normalizedBits = bits - tz;
            if (normalizedBits + productBits >= 64) {
                bignums.add(BigInteger.valueOf(product));
                product = 1L;
                productBits = 0;
            }
            productBits = LongMath.log2(product *= normalizedNum, RoundingMode.FLOOR) + 1;
        }
        if (product > 1L) {
            bignums.add(BigInteger.valueOf(product));
        }
        return BigIntegerMath.listProduct(bignums).shiftLeft(shift);
    }

    static BigInteger listProduct(List<BigInteger> nums) {
        return BigIntegerMath.listProduct(nums, 0, nums.size());
    }

    static BigInteger listProduct(List<BigInteger> nums, int start, int end) {
        switch (end - start) {
            case 0: {
                return BigInteger.ONE;
            }
            case 1: {
                return nums.get(start);
            }
            case 2: {
                return nums.get(start).multiply(nums.get(start + 1));
            }
            case 3: {
                return nums.get(start).multiply(nums.get(start + 1)).multiply(nums.get(start + 2));
            }
        }
        int m4 = end + start >>> 1;
        return BigIntegerMath.listProduct(nums, start, m4).multiply(BigIntegerMath.listProduct(nums, m4, end));
    }

    public static BigInteger binomial(int n4, int k4) {
        int bits;
        MathPreconditions.checkNonNegative("n", n4);
        MathPreconditions.checkNonNegative("k", k4);
        Preconditions.checkArgument(k4 <= n4, "k (%s) > n (%s)", k4, n4);
        if (k4 > n4 >> 1) {
            k4 = n4 - k4;
        }
        if (k4 < LongMath.biggestBinomials.length && n4 <= LongMath.biggestBinomials[k4]) {
            return BigInteger.valueOf(LongMath.binomial(n4, k4));
        }
        BigInteger accum = BigInteger.ONE;
        long numeratorAccum = n4;
        long denominatorAccum = 1L;
        int numeratorBits = bits = LongMath.log2(n4, RoundingMode.CEILING);
        for (int i4 = 1; i4 < k4; ++i4) {
            int p4 = n4 - i4;
            int q4 = i4 + 1;
            if (numeratorBits + bits >= 63) {
                accum = accum.multiply(BigInteger.valueOf(numeratorAccum)).divide(BigInteger.valueOf(denominatorAccum));
                numeratorAccum = p4;
                denominatorAccum = q4;
                numeratorBits = bits;
                continue;
            }
            numeratorAccum *= (long)p4;
            denominatorAccum *= (long)q4;
            numeratorBits += bits;
        }
        return accum.multiply(BigInteger.valueOf(numeratorAccum)).divide(BigInteger.valueOf(denominatorAccum));
    }

    @GwtIncompatible
    static boolean fitsInLong(BigInteger x4) {
        return x4.bitLength() <= 63;
    }

    private BigIntegerMath() {
    }

    @GwtIncompatible
    private static class BigIntegerToDoubleRounder
    extends ToDoubleRounder<BigInteger> {
        static final BigIntegerToDoubleRounder INSTANCE = new BigIntegerToDoubleRounder();

        private BigIntegerToDoubleRounder() {
        }

        @Override
        double roundToDoubleArbitrarily(BigInteger bigInteger) {
            return DoubleUtils.bigToDouble(bigInteger);
        }

        @Override
        int sign(BigInteger bigInteger) {
            return bigInteger.signum();
        }

        @Override
        BigInteger toX(double d5, RoundingMode mode) {
            return DoubleMath.roundToBigInteger(d5, mode);
        }

        @Override
        BigInteger minus(BigInteger a5, BigInteger b5) {
            return a5.subtract(b5);
        }
    }
}

