/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleUtils;
import com.google.common.math.IntMath;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.Longs;
import com.google.common.primitives.UnsignedLongs;
import java.math.RoundingMode;

@GwtCompatible(emulated=true)
public final class LongMath {
    @VisibleForTesting
    static final long MAX_SIGNED_POWER_OF_TWO = 0x4000000000000000L;
    @VisibleForTesting
    static final long MAX_POWER_OF_SQRT2_UNSIGNED = -5402926248376769404L;
    @VisibleForTesting
    static final byte[] maxLog10ForLeadingZeros = new byte[]{19, 18, 18, 18, 18, 17, 17, 17, 16, 16, 16, 15, 15, 15, 15, 14, 14, 14, 13, 13, 13, 12, 12, 12, 12, 11, 11, 11, 10, 10, 10, 9, 9, 9, 9, 8, 8, 8, 7, 7, 7, 6, 6, 6, 6, 5, 5, 5, 4, 4, 4, 3, 3, 3, 3, 2, 2, 2, 1, 1, 1, 0, 0, 0};
    @GwtIncompatible
    @VisibleForTesting
    static final long[] powersOf10 = new long[]{1L, 10L, 100L, 1000L, 10000L, 100000L, 1000000L, 10000000L, 100000000L, 1000000000L, 10000000000L, 100000000000L, 1000000000000L, 10000000000000L, 100000000000000L, 1000000000000000L, 10000000000000000L, 100000000000000000L, 1000000000000000000L};
    @GwtIncompatible
    @VisibleForTesting
    static final long[] halfPowersOf10 = new long[]{3L, 31L, 316L, 3162L, 31622L, 316227L, 3162277L, 31622776L, 316227766L, 3162277660L, 31622776601L, 316227766016L, 3162277660168L, 31622776601683L, 316227766016837L, 3162277660168379L, 31622776601683793L, 316227766016837933L, 3162277660168379331L};
    @VisibleForTesting
    static final long FLOOR_SQRT_MAX_LONG = 3037000499L;
    static final long[] factorials = new long[]{1L, 1L, 2L, 6L, 24L, 120L, 720L, 5040L, 40320L, 362880L, 3628800L, 39916800L, 479001600L, 6227020800L, 87178291200L, 1307674368000L, 20922789888000L, 355687428096000L, 6402373705728000L, 121645100408832000L, 2432902008176640000L};
    static final int[] biggestBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, 3810779, 121977, 16175, 4337, 1733, 887, 534, 361, 265, 206, 169, 143, 125, 111, 101, 94, 88, 83, 79, 76, 74, 72, 70, 69, 68, 67, 67, 66, 66, 66, 66};
    @VisibleForTesting
    static final int[] biggestSimpleBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, 2642246, 86251, 11724, 3218, 1313, 684, 419, 287, 214, 169, 139, 119, 105, 95, 87, 81, 76, 73, 70, 68, 66, 64, 63, 62, 62, 61, 61, 61};
    private static final int SIEVE_30 = -545925251;
    private static final long[][] millerRabinBaseSets = new long[][]{{291830L, 126401071349994536L}, {885594168L, 725270293939359937L, 3569819667048198375L}, {273919523040L, 15L, 7363882082L, 992620450144556L}, {47636622961200L, 2L, 2570940L, 211991001L, 3749873356L}, {7999252175582850L, 2L, 4130806001517L, 149795463772692060L, 186635894390467037L, 3967304179347715805L}, {585226005592931976L, 2L, 123635709730000L, 9233062284813009L, 43835965440333360L, 761179012939631437L, 1263739024124850375L}, {Long.MAX_VALUE, 2L, 325L, 9375L, 28178L, 450775L, 9780504L, 1795265022L}};

    @Beta
    public static long ceilingPowerOfTwo(long x4) {
        MathPreconditions.checkPositive("x", x4);
        if (x4 > 0x4000000000000000L) {
            throw new ArithmeticException(new StringBuilder(70).append("ceilingPowerOfTwo(").append(x4).append(") is not representable as a long").toString());
        }
        return 1L << -Long.numberOfLeadingZeros(x4 - 1L);
    }

    @Beta
    public static long floorPowerOfTwo(long x4) {
        MathPreconditions.checkPositive("x", x4);
        return 1L << 63 - Long.numberOfLeadingZeros(x4);
    }

    public static boolean isPowerOfTwo(long x4) {
        return x4 > 0L & (x4 & x4 - 1L) == 0L;
    }

    @VisibleForTesting
    static int lessThanBranchFree(long x4, long y4) {
        return (int)((x4 - y4 ^ 0xFFFFFFFFFFFFFFFFL ^ 0xFFFFFFFFFFFFFFFFL) >>> 63);
    }

    public static int log2(long x4, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x4);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(LongMath.isPowerOfTwo(x4));
            }
            case DOWN: 
            case FLOOR: {
                return 63 - Long.numberOfLeadingZeros(x4);
            }
            case UP: 
            case CEILING: {
                return 64 - Long.numberOfLeadingZeros(x4 - 1L);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int leadingZeros = Long.numberOfLeadingZeros(x4);
                long cmp = -5402926248376769404L >>> leadingZeros;
                int logFloor = 63 - leadingZeros;
                return logFloor + LongMath.lessThanBranchFree(cmp, x4);
            }
        }
        throw new AssertionError((Object)"impossible");
    }

    @GwtIncompatible
    public static int log10(long x4, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x4);
        int logFloor = LongMath.log10Floor(x4);
        long floorPow = powersOf10[logFloor];
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(x4 == floorPow);
            }
            case DOWN: 
            case FLOOR: {
                return logFloor;
            }
            case UP: 
            case CEILING: {
                return logFloor + LongMath.lessThanBranchFree(floorPow, x4);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                return logFloor + LongMath.lessThanBranchFree(halfPowersOf10[logFloor], x4);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    static int log10Floor(long x4) {
        byte y4 = maxLog10ForLeadingZeros[Long.numberOfLeadingZeros(x4)];
        return y4 - LongMath.lessThanBranchFree(x4, powersOf10[y4]);
    }

    @GwtIncompatible
    public static long pow(long b5, int k4) {
        MathPreconditions.checkNonNegative("exponent", k4);
        if (-2L <= b5 && b5 <= 2L) {
            switch ((int)b5) {
                case 0: {
                    return k4 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (k4 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    return k4 < 64 ? 1L << k4 : 0L;
                }
                case -2: {
                    if (k4 < 64) {
                        return (k4 & 1) == 0 ? 1L << k4 : -(1L << k4);
                    }
                    return 0L;
                }
            }
            throw new AssertionError();
        }
        long accum = 1L;
        while (true) {
            switch (k4) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return accum * b5;
                }
            }
            accum *= (k4 & 1) == 0 ? 1L : b5;
            b5 *= b5;
            k4 >>= 1;
        }
    }

    @GwtIncompatible
    public static long sqrt(long x4, RoundingMode mode) {
        MathPreconditions.checkNonNegative("x", x4);
        if (LongMath.fitsInInt(x4)) {
            return IntMath.sqrt((int)x4, mode);
        }
        long guess = (long)Math.sqrt(x4);
        long guessSquared = guess * guess;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(guessSquared == x4);
                return guess;
            }
            case DOWN: 
            case FLOOR: {
                if (x4 < guessSquared) {
                    return guess - 1L;
                }
                return guess;
            }
            case UP: 
            case CEILING: {
                if (x4 > guessSquared) {
                    return guess + 1L;
                }
                return guess;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                long sqrtFloor = guess - (long)(x4 < guessSquared ? 1 : 0);
                long halfSquare = sqrtFloor * sqrtFloor + sqrtFloor;
                return sqrtFloor + (long)LongMath.lessThanBranchFree(halfSquare, x4);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static long divide(long p4, long q4, RoundingMode mode) {
        boolean increment;
        Preconditions.checkNotNull(mode);
        long div = p4 / q4;
        long rem = p4 - q4 * div;
        if (rem == 0L) {
            return div;
        }
        int signum = 1 | (int)((p4 ^ q4) >> 63);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(rem == 0L);
            }
            case DOWN: {
                increment = false;
                break;
            }
            case UP: {
                increment = true;
                break;
            }
            case CEILING: {
                increment = signum > 0;
                break;
            }
            case FLOOR: {
                increment = signum < 0;
                break;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                long absRem = Math.abs(rem);
                long cmpRemToHalfDivisor = absRem - (Math.abs(q4) - absRem);
                if (cmpRemToHalfDivisor == 0L) {
                    increment = mode == RoundingMode.HALF_UP | mode == RoundingMode.HALF_EVEN & (div & 1L) != 0L;
                    break;
                }
                increment = cmpRemToHalfDivisor > 0L;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return increment ? div + (long)signum : div;
    }

    @GwtIncompatible
    public static int mod(long x4, int m4) {
        return (int)LongMath.mod(x4, (long)m4);
    }

    @GwtIncompatible
    public static long mod(long x4, long m4) {
        if (m4 <= 0L) {
            throw new ArithmeticException("Modulus must be positive");
        }
        long result = x4 % m4;
        return result >= 0L ? result : result + m4;
    }

    public static long gcd(long a5, long b5) {
        MathPreconditions.checkNonNegative("a", a5);
        MathPreconditions.checkNonNegative("b", b5);
        if (a5 == 0L) {
            return b5;
        }
        if (b5 == 0L) {
            return a5;
        }
        int aTwos = Long.numberOfTrailingZeros(a5);
        a5 >>= aTwos;
        int bTwos = Long.numberOfTrailingZeros(b5);
        b5 >>= bTwos;
        while (a5 != b5) {
            long delta = a5 - b5;
            long minDeltaOrZero = delta & delta >> 63;
            a5 = delta - minDeltaOrZero - minDeltaOrZero;
            b5 += minDeltaOrZero;
            a5 >>= Long.numberOfTrailingZeros(a5);
        }
        return a5 << Math.min(aTwos, bTwos);
    }

    @GwtIncompatible
    public static long checkedAdd(long a5, long b5) {
        long result = a5 + b5;
        MathPreconditions.checkNoOverflow((a5 ^ b5) < 0L | (a5 ^ result) >= 0L, "checkedAdd", a5, b5);
        return result;
    }

    @GwtIncompatible
    public static long checkedSubtract(long a5, long b5) {
        long result = a5 - b5;
        MathPreconditions.checkNoOverflow((a5 ^ b5) >= 0L | (a5 ^ result) >= 0L, "checkedSubtract", a5, b5);
        return result;
    }

    public static long checkedMultiply(long a5, long b5) {
        int leadingZeros = Long.numberOfLeadingZeros(a5) + Long.numberOfLeadingZeros(a5 ^ 0xFFFFFFFFFFFFFFFFL) + Long.numberOfLeadingZeros(b5) + Long.numberOfLeadingZeros(b5 ^ 0xFFFFFFFFFFFFFFFFL);
        if (leadingZeros > 65) {
            return a5 * b5;
        }
        MathPreconditions.checkNoOverflow(leadingZeros >= 64, "checkedMultiply", a5, b5);
        MathPreconditions.checkNoOverflow(a5 >= 0L | b5 != Long.MIN_VALUE, "checkedMultiply", a5, b5);
        long result = a5 * b5;
        MathPreconditions.checkNoOverflow(a5 == 0L || result / a5 == b5, "checkedMultiply", a5, b5);
        return result;
    }

    @GwtIncompatible
    public static long checkedPow(long b5, int k4) {
        MathPreconditions.checkNonNegative("exponent", k4);
        if (b5 >= -2L & b5 <= 2L) {
            switch ((int)b5) {
                case 0: {
                    return k4 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (k4 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    MathPreconditions.checkNoOverflow(k4 < 63, "checkedPow", b5, (long)k4);
                    return 1L << k4;
                }
                case -2: {
                    MathPreconditions.checkNoOverflow(k4 < 64, "checkedPow", b5, (long)k4);
                    return (k4 & 1) == 0 ? 1L << k4 : -1L << k4;
                }
            }
            throw new AssertionError();
        }
        long accum = 1L;
        while (true) {
            switch (k4) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return LongMath.checkedMultiply(accum, b5);
                }
            }
            if ((k4 & 1) != 0) {
                accum = LongMath.checkedMultiply(accum, b5);
            }
            if ((k4 >>= 1) <= 0) continue;
            MathPreconditions.checkNoOverflow(-3037000499L <= b5 && b5 <= 3037000499L, "checkedPow", b5, (long)k4);
            b5 *= b5;
        }
    }

    @Beta
    public static long saturatedAdd(long a5, long b5) {
        long naiveSum;
        if ((a5 ^ b5) < 0L | (a5 ^ (naiveSum = a5 + b5)) >= 0L) {
            return naiveSum;
        }
        return Long.MAX_VALUE + (naiveSum >>> 63 ^ 1L);
    }

    @Beta
    public static long saturatedSubtract(long a5, long b5) {
        long naiveDifference;
        if ((a5 ^ b5) >= 0L | (a5 ^ (naiveDifference = a5 - b5)) >= 0L) {
            return naiveDifference;
        }
        return Long.MAX_VALUE + (naiveDifference >>> 63 ^ 1L);
    }

    @Beta
    public static long saturatedMultiply(long a5, long b5) {
        int leadingZeros = Long.numberOfLeadingZeros(a5) + Long.numberOfLeadingZeros(a5 ^ 0xFFFFFFFFFFFFFFFFL) + Long.numberOfLeadingZeros(b5) + Long.numberOfLeadingZeros(b5 ^ 0xFFFFFFFFFFFFFFFFL);
        if (leadingZeros > 65) {
            return a5 * b5;
        }
        long limit = Long.MAX_VALUE + ((a5 ^ b5) >>> 63);
        if (leadingZeros < 64 | a5 < 0L & b5 == Long.MIN_VALUE) {
            return limit;
        }
        long result = a5 * b5;
        if (a5 == 0L || result / a5 == b5) {
            return result;
        }
        return limit;
    }

    @Beta
    public static long saturatedPow(long b5, int k4) {
        MathPreconditions.checkNonNegative("exponent", k4);
        if (b5 >= -2L & b5 <= 2L) {
            switch ((int)b5) {
                case 0: {
                    return k4 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (k4 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    if (k4 >= 63) {
                        return Long.MAX_VALUE;
                    }
                    return 1L << k4;
                }
                case -2: {
                    if (k4 >= 64) {
                        return Long.MAX_VALUE + (long)(k4 & 1);
                    }
                    return (k4 & 1) == 0 ? 1L << k4 : -1L << k4;
                }
            }
            throw new AssertionError();
        }
        long accum = 1L;
        long limit = Long.MAX_VALUE + (b5 >>> 63 & (long)(k4 & 1));
        while (true) {
            switch (k4) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return LongMath.saturatedMultiply(accum, b5);
                }
            }
            if ((k4 & 1) != 0) {
                accum = LongMath.saturatedMultiply(accum, b5);
            }
            if ((k4 >>= 1) <= 0) continue;
            if (-3037000499L > b5 | b5 > 3037000499L) {
                return limit;
            }
            b5 *= b5;
        }
    }

    @GwtIncompatible
    public static long factorial(int n4) {
        MathPreconditions.checkNonNegative("n", n4);
        return n4 < factorials.length ? factorials[n4] : Long.MAX_VALUE;
    }

    public static long binomial(int n4, int k4) {
        MathPreconditions.checkNonNegative("n", n4);
        MathPreconditions.checkNonNegative("k", k4);
        Preconditions.checkArgument(k4 <= n4, "k (%s) > n (%s)", k4, n4);
        if (k4 > n4 >> 1) {
            k4 = n4 - k4;
        }
        switch (k4) {
            case 0: {
                return 1L;
            }
            case 1: {
                return n4;
            }
        }
        if (n4 < factorials.length) {
            return factorials[n4] / (factorials[k4] * factorials[n4 - k4]);
        }
        if (k4 >= biggestBinomials.length || n4 > biggestBinomials[k4]) {
            return Long.MAX_VALUE;
        }
        if (k4 < biggestSimpleBinomials.length && n4 <= biggestSimpleBinomials[k4]) {
            long result = n4--;
            for (int i4 = 2; i4 <= k4; ++i4) {
                result *= (long)n4;
                result /= (long)i4;
                --n4;
            }
            return result;
        }
        int nBits = LongMath.log2(n4, RoundingMode.CEILING);
        long result = 1L;
        long numerator = n4--;
        long denominator = 1L;
        int numeratorBits = nBits;
        int i5 = 2;
        while (i5 <= k4) {
            if (numeratorBits + nBits < 63) {
                numerator *= (long)n4;
                denominator *= (long)i5;
                numeratorBits += nBits;
            } else {
                result = LongMath.multiplyFraction(result, numerator, denominator);
                numerator = n4;
                denominator = i5;
                numeratorBits = nBits;
            }
            ++i5;
            --n4;
        }
        return LongMath.multiplyFraction(result, numerator, denominator);
    }

    static long multiplyFraction(long x4, long numerator, long denominator) {
        if (x4 == 1L) {
            return numerator / denominator;
        }
        long commonDivisor = LongMath.gcd(x4, denominator);
        return (x4 /= commonDivisor) * (numerator / (denominator /= commonDivisor));
    }

    static boolean fitsInInt(long x4) {
        return (long)((int)x4) == x4;
    }

    public static long mean(long x4, long y4) {
        return (x4 & y4) + ((x4 ^ y4) >> 1);
    }

    @GwtIncompatible
    @Beta
    public static boolean isPrime(long n4) {
        if (n4 < 2L) {
            MathPreconditions.checkNonNegative("n", n4);
            return false;
        }
        if (n4 < 66L) {
            long mask = 722865708377213483L;
            return (mask >> (int)n4 - 2 & 1L) != 0L;
        }
        if ((0xDF75D77D & 1 << (int)(n4 % 30L)) != 0) {
            return false;
        }
        if (n4 % 7L == 0L || n4 % 11L == 0L || n4 % 13L == 0L) {
            return false;
        }
        if (n4 < 289L) {
            return true;
        }
        for (long[] baseSet : millerRabinBaseSets) {
            if (n4 > baseSet[0]) continue;
            for (int i4 = 1; i4 < baseSet.length; ++i4) {
                if (MillerRabinTester.test(baseSet[i4], n4)) continue;
                return false;
            }
            return true;
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static double roundToDouble(long x4, RoundingMode mode) {
        double roundArbitrarily = x4;
        long roundArbitrarilyAsLong = (long)roundArbitrarily;
        int cmpXToRoundArbitrarily = roundArbitrarilyAsLong == Long.MAX_VALUE ? -1 : Longs.compare(x4, roundArbitrarilyAsLong);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(cmpXToRoundArbitrarily == 0);
                return roundArbitrarily;
            }
            case FLOOR: {
                return cmpXToRoundArbitrarily >= 0 ? roundArbitrarily : DoubleUtils.nextDown(roundArbitrarily);
            }
            case CEILING: {
                return cmpXToRoundArbitrarily <= 0 ? roundArbitrarily : Math.nextUp(roundArbitrarily);
            }
            case DOWN: {
                if (x4 >= 0L) {
                    return cmpXToRoundArbitrarily >= 0 ? roundArbitrarily : DoubleUtils.nextDown(roundArbitrarily);
                }
                return cmpXToRoundArbitrarily <= 0 ? roundArbitrarily : Math.nextUp(roundArbitrarily);
            }
            case UP: {
                if (x4 >= 0L) {
                    return cmpXToRoundArbitrarily <= 0 ? roundArbitrarily : Math.nextUp(roundArbitrarily);
                }
                return cmpXToRoundArbitrarily >= 0 ? roundArbitrarily : DoubleUtils.nextDown(roundArbitrarily);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int diff;
                long roundCeiling;
                double roundCeilingAsDouble;
                long roundFloor;
                double roundFloorAsDouble;
                if (cmpXToRoundArbitrarily >= 0) {
                    roundFloorAsDouble = roundArbitrarily;
                    roundFloor = roundArbitrarilyAsLong;
                    roundCeilingAsDouble = Math.nextUp(roundArbitrarily);
                    roundCeiling = (long)Math.ceil(roundCeilingAsDouble);
                } else {
                    roundCeilingAsDouble = roundArbitrarily;
                    roundCeiling = roundArbitrarilyAsLong;
                    roundFloorAsDouble = DoubleUtils.nextDown(roundArbitrarily);
                    roundFloor = (long)Math.floor(roundFloorAsDouble);
                }
                long deltaToFloor = x4 - roundFloor;
                long deltaToCeiling = roundCeiling - x4;
                if (roundCeiling == Long.MAX_VALUE) {
                    ++deltaToCeiling;
                }
                if ((diff = Longs.compare(deltaToFloor, deltaToCeiling)) < 0) {
                    return roundFloorAsDouble;
                }
                if (diff > 0) {
                    return roundCeilingAsDouble;
                }
                switch (mode) {
                    case HALF_EVEN: {
                        return (DoubleUtils.getSignificand(roundFloorAsDouble) & 1L) == 0L ? roundFloorAsDouble : roundCeilingAsDouble;
                    }
                    case HALF_DOWN: {
                        return x4 >= 0L ? roundFloorAsDouble : roundCeilingAsDouble;
                    }
                    case HALF_UP: {
                        return x4 >= 0L ? roundCeilingAsDouble : roundFloorAsDouble;
                    }
                }
                throw new AssertionError((Object)"impossible");
            }
        }
        throw new AssertionError((Object)"impossible");
    }

    private LongMath() {
    }

    private static enum MillerRabinTester {
        SMALL{

            @Override
            long mulMod(long a5, long b5, long m4) {
                return a5 * b5 % m4;
            }

            @Override
            long squareMod(long a5, long m4) {
                return a5 * a5 % m4;
            }
        }
        ,
        LARGE{

            private long plusMod(long a5, long b5, long m4) {
                return a5 >= m4 - b5 ? a5 + b5 - m4 : a5 + b5;
            }

            private long times2ToThe32Mod(long a5, long m4) {
                int shift;
                int remainingPowersOf2 = 32;
                do {
                    shift = Math.min(remainingPowersOf2, Long.numberOfLeadingZeros(a5));
                    a5 = UnsignedLongs.remainder(a5 << shift, m4);
                } while ((remainingPowersOf2 -= shift) > 0);
                return a5;
            }

            @Override
            long mulMod(long a5, long b5, long m4) {
                long aHi = a5 >>> 32;
                long bHi = b5 >>> 32;
                long aLo = a5 & 0xFFFFFFFFL;
                long bLo = b5 & 0xFFFFFFFFL;
                long result = this.times2ToThe32Mod(aHi * bHi, m4);
                if ((result += aHi * bLo) < 0L) {
                    result = UnsignedLongs.remainder(result, m4);
                }
                result += aLo * bHi;
                result = this.times2ToThe32Mod(result, m4);
                return this.plusMod(result, UnsignedLongs.remainder(aLo * bLo, m4), m4);
            }

            @Override
            long squareMod(long a5, long m4) {
                long aHi = a5 >>> 32;
                long aLo = a5 & 0xFFFFFFFFL;
                long result = this.times2ToThe32Mod(aHi * aHi, m4);
                long hiLo = aHi * aLo * 2L;
                if (hiLo < 0L) {
                    hiLo = UnsignedLongs.remainder(hiLo, m4);
                }
                result += hiLo;
                result = this.times2ToThe32Mod(result, m4);
                return this.plusMod(result, UnsignedLongs.remainder(aLo * aLo, m4), m4);
            }
        };


        static boolean test(long base, long n4) {
            return (n4 <= 3037000499L ? SMALL : LARGE).testWitness(base, n4);
        }

        abstract long mulMod(long var1, long var3, long var5);

        abstract long squareMod(long var1, long var3);

        private long powMod(long a5, long p4, long m4) {
            long res = 1L;
            while (p4 != 0L) {
                if ((p4 & 1L) != 0L) {
                    res = this.mulMod(res, a5, m4);
                }
                a5 = this.squareMod(a5, m4);
                p4 >>= 1;
            }
            return res;
        }

        private boolean testWitness(long base, long n4) {
            int r4 = Long.numberOfTrailingZeros(n4 - 1L);
            long d5 = n4 - 1L >> r4;
            if ((base %= n4) == 0L) {
                return true;
            }
            long a5 = this.powMod(base, d5, n4);
            if (a5 == 1L) {
                return true;
            }
            int j4 = 0;
            while (a5 != n4 - 1L) {
                if (++j4 == r4) {
                    return false;
                }
                a5 = this.squareMod(a5, n4);
            }
            return true;
        }
    }
}

